﻿/**
 * 🎹 純 JavaScript 虛擬鍵盤 - 無外部依賴
 * 點擊 btnKeyboard 顯示，Enter 送出文字到聚焦的輸入框
 */

(function () {
    'use strict';

    if (window.keyboardInjected) {
        console.log('🎹 Keyboard already injected');
        return;
    }
    window.keyboardInjected = true;

    // ✅ 鍵盤佈局定義
    const KEYBOARD_LAYOUTS = {
        default: [
            ['1', '2', '3', '4', '5', '6', '7', '8', '9', '0', 'Bksp'],
            ['q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p'],
            ['a', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l'],
            ['Shift', 'z', 'x', 'c', 'v', 'b', 'n', 'm'],
            ['Space', 'Enter']
        ],
        shift: [
            ['!', '@', '#', '$', '%', '^', '&', '*', '(', ')', 'Bksp'],
            ['Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P'],
            ['A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L'],
            ['Shift', 'Z', 'X', 'C', 'V', 'B', 'N', 'M'],
            ['Space', 'Enter']
        ]
    };

    const BUTTON_DISPLAY = {
        'Bksp': '⌫ Backspace',
        'Enter': '↵ Enter',
        'Space': 'Space',
        'Shift': '⇧ Shift'
    };

    // ✅ 創建鍵盤樣式
    const keyboardStyle = document.createElement('style');
    keyboardStyle.textContent = `
        #keyboard-container {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            z-index: 9999;
            background: #fff;
            box-shadow: 0 -5px 20px rgba(0,0,0,0.3);
            padding: 10px;
            display: none;
            font-family: Arial, sans-serif;
        }
        #keyboard-container.show {
            display: block;
        }
        .keyboard-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 10px;
            background: #f0f0f0;
            border-radius: 5px;
            margin-bottom: 10px;
        }
        .keyboard-header-title {
            font-weight: bold;
            font-size: 18px;
        }
        .keyboard-close {
            background: #ff5555;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            font-weight: bold;
            transition: all 0.2s;
        }
        .keyboard-close:hover {
            background: #ff0000;
        }
        .keyboard-close.focused {
            background: #cc0000 !important;
            box-shadow: 0 0 0 4px #ffcc00 !important;
            transform: scale(1.1) !important;
        }
        .keyboard-content {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        .keyboard-row {
            display: flex;
            gap: 6px;
            justify-content: center;
            flex-wrap: wrap;
        }
        .keyboard-key {
            background: #e0e0e0;
            border: 1px solid #bdbdbd;
            border-radius: 4px;
            padding: 12px 16px;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.2s;
            min-width: 40px;
            text-align: center;
            user-select: none;
        }
        .keyboard-key:hover {
            background: #d0d0d0;
        }
        .keyboard-key:active {
            background: #a0a0a0;
            color: white;
        }
        .keyboard-key.hg-focused {
            background: linear-gradient(to bottom, #4da6ff 0%, #0066cc 100%);
            color: white;
            box-shadow: 0 0 0 4px #ffcc00, 0 0 20px rgba(0, 102, 204, 0.5);
            transform: scale(1.1);
            z-index: 10;
        }
        .keyboard-key.special {
            background: #90caf9;
            min-width: 60px;
        }
        .keyboard-key.space-key {
            flex: 1;
            min-width: 200px;
        }
        .keyboard-key.enter-key {
            flex: 1;
            background: #4caf50;
            color: white;
            min-width: 100px;
        }
    `;
    document.head.appendChild(keyboardStyle);

    // ✅ 初始化鍵盤
    function initKeyboard() {
        console.log('🎹 Initializing pure JavaScript keyboard...');

        const container = document.createElement('div');
        container.id = 'keyboard-container';

        let currentLayout = 'default';
        let currentInput = null;
        let isKeyboardActive = false;
        let currentFocusIndex = -1;
        let allFocusableElements = [];

        // ✅ 創建鍵盤 HTML - 使用 DOM API 而不是 innerHTML
        function renderKeyboard() {
            const layout = KEYBOARD_LAYOUTS[currentLayout];
            
            // 清空容器
            container.textContent = '';

            // 創建 header
            const header = document.createElement('div');
            header.className = 'keyboard-header';

            const title = document.createElement('span');
            title.className = 'keyboard-header-title';
            title.textContent = '🎹 虛擬鍵盤';

            const closeBtn = document.createElement('button');
            closeBtn.className = 'keyboard-close';
            closeBtn.id = 'keyboard-close-btn';
            closeBtn.textContent = '✕ 關閉';

            header.appendChild(title);
            header.appendChild(closeBtn);
            container.appendChild(header);

            // 創建鍵盤內容
            const content = document.createElement('div');
            content.className = 'keyboard-content';

            layout.forEach(row => {
                const rowDiv = document.createElement('div');
                rowDiv.className = 'keyboard-row';

                row.forEach(key => {
                    const displayText = BUTTON_DISPLAY[key] || key;
                    let keyClass = 'keyboard-key';
                    if (key === 'Space') keyClass += ' special space-key';
                    else if (key === 'Enter') keyClass += ' special enter-key';
                    else if (key === 'Shift' || key === 'Bksp') keyClass += ' special';

                    const btn = document.createElement('button');
                    btn.className = keyClass;
                    btn.dataset.key = key;
                    btn.textContent = displayText;

                    rowDiv.appendChild(btn);
                });

                content.appendChild(rowDiv);
            });

            container.appendChild(content);
        }

        // ✅ 首次渲染
        renderKeyboard();

        if (!document.body) {
            console.error('❌ document.body not available');
            setTimeout(initKeyboard, 100);
            return;
        }

        document.body.appendChild(container);
        console.log('✅ Keyboard container added to DOM');

        // ✅ 更新可聚焦元素列表
        function updateFocusableElements() {
            const keyButtons = Array.from(container.querySelectorAll('.keyboard-key'));
            const closeBtn = container.querySelector('#keyboard-close-btn');
            allFocusableElements = [...keyButtons, closeBtn];
        }

        // ✅ 高亮按鈕
        function highlightButton(element) {
            container.querySelectorAll('.keyboard-key, .keyboard-close').forEach(el => {
                el.classList.remove('hg-focused');
            });
            if (element) {
                element.classList.add('hg-focused');
            }
        }

        // ✅ 處理鍵盤按鈕點擊
        container.addEventListener('click', (e) => {
            const keyBtn = e.target.closest('.keyboard-key');
            if (keyBtn) {
                const key = keyBtn.dataset.key;
                handleKeyPress(key);
                return;
            }

            const closeBtn = e.target.closest('#keyboard-close-btn');
            if (closeBtn) {
                window.hideVirtualKeyboard();
            }
        });

        // ✅ 處理按鍵邏輯
        function handleKeyPress(key) {
            console.log('🔘 Key pressed:', key);

            if (key === 'Bksp') {
                if (currentInput) {
                    const val = currentInput.value;
                    currentInput.value = val.slice(0, -1);
                    currentInput.dispatchEvent(new Event('input', { bubbles: true }));
                }
            } else if (key === 'Enter') {
                if (currentInput) {
                    currentInput.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', keyCode: 13, bubbles: true }));
                    const form = currentInput.closest('form');
                    if (form) {
                        form.dispatchEvent(new Event('submit', { bubbles: true }));
                    }
                    console.log('✅ Enter pressed');
                }
            } else if (key === 'Space') {
                if (currentInput) {
                    currentInput.value += ' ';
                    currentInput.dispatchEvent(new Event('input', { bubbles: true }));
                }
            } else if (key === 'Shift') {
                currentLayout = currentLayout === 'default' ? 'shift' : 'default';
                renderKeyboard();
                updateFocusableElements();
                console.log('✅ Shift toggled, layout:', currentLayout);
            } else {
                if (currentInput) {
                    currentInput.value += key;
                    currentInput.dispatchEvent(new Event('input', { bubbles: true }));
                }
            }
        }

        // ✅ Tab 鍵處理
        document.addEventListener('keydown', (e) => {
            if (!isKeyboardActive) return;

            if (e.key === 'Tab') {
                e.preventDefault();
                e.stopPropagation();

                updateFocusableElements();

                if (e.shiftKey) {
                    currentFocusIndex--;
                    if (currentFocusIndex < 0) {
                        currentFocusIndex = allFocusableElements.length - 1;
                    }
                } else {
                    currentFocusIndex++;
                    if (currentFocusIndex >= allFocusableElements.length) {
                        currentFocusIndex = 0;
                    }
                }

                const element = allFocusableElements[currentFocusIndex];
                highlightButton(element);
                element.focus();
                console.log(`${e.shiftKey ? '⬅️' : '➡️'} Tab: index ${currentFocusIndex}`);
            }
        }, true);

        // ✅ 追蹤輸入框聚焦
        document.addEventListener('focusin', (e) => {
            if (e.target.matches('input[type="text"], input[type="email"], input[type="search"], textarea')) {
                currentInput = e.target;
                console.log('📝 Input focused');
            }
        });

        // ✅ 全域函式
        window.showVirtualKeyboard = function () {
            container.classList.add('show');
            isKeyboardActive = true;
            currentFocusIndex = -1;
            updateFocusableElements();
            console.log('✅ Keyboard shown');

            if (!currentInput) {
                const firstInput = document.querySelector('input[type="text"], input[type="email"], textarea');
                if (firstInput) {
                    firstInput.focus();
                    currentInput = firstInput;
                }
            }

            setTimeout(() => {
                const firstKey = container.querySelector('.keyboard-key');
                if (firstKey) {
                    currentFocusIndex = 0;
                    highlightButton(firstKey);
                    firstKey.focus();
                }
            }, 150);
        };

        window.hideVirtualKeyboard = function () {
            container.classList.remove('show');
            isKeyboardActive = false;
            currentFocusIndex = -1;
            container.querySelectorAll('.keyboard-key, .keyboard-close').forEach(el => {
                el.classList.remove('hg-focused');
            });
            console.log('✅ Keyboard hidden');
        };

        window.toggleVirtualKeyboard = function () {
            if (container.classList.contains('show')) {
                window.hideVirtualKeyboard();
            } else {
                window.showVirtualKeyboard();
            }
        };

        window.isVirtualKeyboardActive = function () {
            return isKeyboardActive;
        };

        window.handleVirtualClick = function () {
            if (isKeyboardActive) {
                const focusedBtn = container.querySelector('.keyboard-key.hg-focused, .keyboard-close.hg-focused');
                if (focusedBtn) {
                    focusedBtn.click();
                    return JSON.stringify({ success: true, action: 'key_pressed' });
                }
                return JSON.stringify({ success: false, reason: 'No key focused' });
            }

            if (document.activeElement && document.activeElement !== document.body) {
                document.activeElement.click();
                return JSON.stringify({ success: true, action: 'clicked_element' });
            }

            return JSON.stringify({ success: false, reason: 'No element focused' });
        };

        console.log('✅ Pure JavaScript keyboard initialized successfully');
    }

    // ✅ DOM 準備完成後初始化
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initKeyboard);
    } else {
        initKeyboard();
    }
})();